
{*******************************************************}
{                                                       }
{        AdvFocusEdit, AdvNumberEdit components         }
{                                                       }
{       Copyright (c) 2002 Tams Molnr, Hungary        }
{             e-mail: mot@merkantil.hu                  }
{                                                       }
{  Use it free. Please, send your observations,         }
{  new ideas, changes                                   }
{                                                       }
{*******************************************************}

{ v1.3 All real types changed to double                 }
{      New property TAdvNumberEdit.Precision            }
{ v1.2 Handling of ParentColor, ParentFont fixed.       }
{ v1.1 Redundant parts removed.                         }
{      Split functions to TAdvFocusEdit (inherited from }
{      TEdit, + base select and focus functions) and    }
{      TAdvNumberEdit                                   }
{ v1.0 All functions tested with demo programs.         }

unit AdvEdits;
interface

uses
  Windows, Classes, Messages, Clipbrd, StdCtrls, Controls,
  Graphics, SysUtils;

{
  TAdvFocusEdit
    TAdvFocusEdit inherited from TEdit
    * it can set alignment of text,
    * Enter and/or cursor arrows can do like key (Shift+)Tab
    * it can define background and font color and style
      when control gets the focus
    * it can hold the focus on control with help of event OnExitQuery

  New properties of TAdvFocusEdit

    Alignment: TAlignment           default taLeftJustify
    ---------------------
      Use Alignment to change the way the text is formatted by the
      AdvFocusEdit control.
      Alignment can take one of the following values:

      Value            Meaning

      taLeftJustify    Align text to the left side of the control
      taCenter         Center text horizontally in the control.
      taRightJustify   Align text to the right side of the control.


    AutoSelectOnClick: boolean      default false
    --------------------------
      By true value of AutoSelect the AdvFocusEdit control selects all the text
      when gets focus by mouse click. Use AutoSelect when the user is more
      likely to replace the text in the AdvFocusEdit control than to append to it.


    EnterAsTab: boolean             default false
    -------------------
      By true value of EnterAsTab the Enter key behaves like Tab key. The
      (Shift+)Enter pressing focus the next (previous) control in the form
      otherwise Enter key is an invalid key press.


    FocusColor: TColor              default clWindow
    ------------------
      Determines the background color of the AdvFocusEdit component when gets
      focus. If the value of FocusColor is same as value of Color property
      it hasn't shown the effect.


    FocusFont: TFont                default Font
    ----------------
      Determines the displaying text font characteristics when gets focus.
      If the value of FocusFont is same as value of Font property it hasn't
      shown the effect.


    UpDownAsTab: boolean            default false
    --------------------
      By true value of UpDownAsTab the Up and Down cursor keys behave like
      Tab key. The Down (Up) pressing focus the next (previous) control in
      the form otherwise Up and Down is same as Left and Right cursor keys.


  New event of TAdvFocusEdit

    OnExitQuery: TExitQueryEvent
    ----------------------------
      Occurs before the input focus shifts away from one control to another.
      Focus changes and OnExit event is called only if CanExit parameter is
      true (default) otherwise focus stays on current AdvFocusEdit component
      (if Enabled and Visible properties are set).





  TAdvNumberEdit
    TAdvNumberEdit inherited from AdvFocusEdit
    * it accepts only numbers (decimal point and sign as well)
    * adjustable whole and decimal length
    * thousand grouping is adjustable
    * min and max checking with event OnBoundError
    * free adjustable hotkey definitions for base operation (+,-,*,/)

  New properties of TAdvNumberEdit

  AdvNumberEdit is descendant of AdvFocusEdit, see above the properties and
  events, additional properties and event see below:


    BeepOnError: boolean            default true
    --------------------
      By true value of BeepOnError the AdvNumberEdit control make a beep when
      the user presses an invalid key otherwise none of key presses make sound.


    CanEmpty: boolean               default false
    -----------------
      By true value of CanEmpty the text can be empty (in this case the value
      is 0). The empty value is never out of bounds by min-max checking.
      By false value text can't be empty, empty text changes immediately to 0.


    DecimalLen: byte                default 0
    ----------------
      Determines the length of the fractional part of the number in AdvNumberEdit
      control. Modification of the DecimalLen property truncates the fractional
      part of the number if needed.


    DecimalSeparator: char          default '.'
    ----------------------
      Determines the character used to separate the decimal portion of a
      number from its integer portion. The sign can be arbitrary character
      except the followings: numbers (0-9), minus ('-') and the actual
      thousand separator sign.


    EditorMode: boolean             default false
    -------------------
      By true value of EditorMode the number formatting (thousand grouping)
      is hidden while editing the number otherwise the number formatting is
      immediately shows on change of the number.


    FunctionList: TFunctionList
    ---------------------------
      Use FunctionList to assign a key combination (with their Shift state
      and virtual key code) to a base math operation (+,-,*,/).
      For example, if the key 'K' assigned to multiplication with 1000,
      it changes the value 2 to 2000 if user presses the key 'K'.


    InvertSign: boolean             default false
    -------------------
      By true value of InvertSign the minus sign pressing changes the sign of
      the number to positive from negative and vice versa otherwise minus sign
      pressing is valid only the position of the first character in the control.

      Note: minus sign press is an invalid key press when minimum checking is
      on and minimum value is greater than or equal 0.


    Min: double
    ---------
      Min determines the minimum value of the number if MinChecking property
      has true value otherwise it has no function.


    MinChecking: boolean            default false
    --------------------
      By true value of MinChecking the Min property determines the minimum
      value of the number otherwise it has no minimum checking.
      In fact the minimum value is given by GetMin(Min,Max) function, that's
      why it doesn't make any problem if the value of Min is greater than
      value of Max.


    Max: double
    ---------
      Max determines the maximum value of the number if MaxChecking property
      has true value otherwise it has no function.


    MaxChecking: boolean            default false
    --------------------
      By true value of MaxChecking the Max property determines the maximum
      value of the number otherwise it has no maximum checking.
      In fact the maximum value is given by GetMax(Min,Max) function, that's
      why it doesn't make any problem if the value of Max is less than value
      of Max.


    Precision: byte                  default 0
    ---------------
      Determines the length of the permanent visible decimal part of the
      number in AdvNumberEdit control.
      Value 0 means general format (meaningless zeros truncated).
      Max value of property Precision is value of property DecimalLen.


    ThousandGrouping: boolean       default true
    -------------------------
      By true value of ThousandGrouping the large numbers is separated into
      their "thousands" components otherwise the number has no thousand
      separation.


    ThousandSeparator: char         default ','
    -----------------------
      Determines the thousand separator sign. The sign can be arbitrary
      character except the followings: numbers (0-9), dot ('.'), minus
      ('-') and the actual decimal separator sign. It has no meaning if
      ThousandGrouping property is false.


    Value: double
    -----------
      Numeric value of the Text property, it can be read/write. Of course
      the changing of this property also changes the Text property.


    WholeLen: byte                  default 8
    --------------
      Determines the length of the whole part of the number in AdvNumberEdit
      control. Modification of the WholeLen property truncates the whole
      part of number if needed.


  New event of TAdvNumberEdit

    OnBoundError: TBoundErrorEvent
    ------------------------------
      Occurs when the value of the property Value violates the bounds
      determinate by properties Min and Max if the properties MinChecking
      and/or MaxChecking are set.
      Parameter MinViolation is true when minimum bound is violated
      otherwise the maximum bound is violated.
      Parameter BoundValue gives the terminal value of the bound.

      Note: when property CanEmpty is set and property Text is empty ('')
      the OnBoundError event never occurs because the empty value is within
      all bounds.

}

const
  OprStr = '+-*/';

type
  TOperation = (opAdd, opSub, opMul, opDiv);
  TOperations = set of TOperation;

type
  TFunctionList = class(TStringList)
  private
    function GetKey(Index: integer) : integer;
    function GetOperation(Index: integer) : TOperation;
    function GetShift(Index: integer) : TShiftState;
    function GetValue(Index: integer) : double;
    procedure InsertNew(Index, ItemNo: integer; NewItem: string);
    procedure SetKey(Index: integer; AValue: integer);
    procedure SetOperation(Index: integer; AValue: TOperation);
    procedure SetShift(Index: integer; AValue: TShiftState);
    procedure SetValue(Index: integer; AValue: double);
  public
    procedure GetItem(Index: integer; var Shift: TShiftState; var Key: word;
        var Operation: TOperation; var Value: double);
    property Key[Index: integer] : integer read GetKey write SetKey;
    property Operation[Index: integer] : TOperation read GetOperation write SetOperation;
    property Shift[Index: integer] : TShiftState read GetShift write SetShift;
    property Value[Index: integer] : double read GetValue write SetValue;
  end;

type
  TExitQueryEvent = procedure(Sender: TObject; var CanExit: boolean) of object;
  TBoundErrorEvent = procedure (Sender: TObject; MinViolation: boolean;
    BoundValue: double) of object;

type
  EBoundError = class(Exception);

type
  TAdvFocusEdit = class(TEdit)
  private
    FOrigColor: TColor;
    FOrigFont: TFont;
    FOrigParentColor: boolean;
    FOrigParentFont: boolean;
    FAlignment: TAlignment;
    FAutoSelectOnClick: boolean;
    FEnterAsTab: boolean;
    FFocusColor: TColor;
    FFocusFont: TFont;
    FUpDownAsTab: boolean;
    FOnExitQuery: TExitQueryEvent;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure ChangeControl(Back: boolean);
    procedure SetAlignment(AValue: TAlignment);
    procedure SetFocusColor(AValue: TColor);
    procedure SetFocusFont(AValue: TFont);
    procedure Click; override;
    procedure DoEnter; override;
    procedure DoExitQuery(var CanExit: boolean); dynamic;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property AutoSelectOnClick: boolean read FAutoSelectOnClick write FAutoSelectOnClick;
    property EnterAsTab: boolean read FEnterAsTab write FEnterAsTab;
    property FocusColor: TColor read FFocusColor write SetFocusColor default clWindow;
    property FocusFont: TFont read FFocusFont write SetFocusFont;
    property UpDownAsTab: boolean read FUpDownAsTab write FUpDownAsTab;
    property OnExitQuery: TExitQueryEvent read FOnExitQuery write FOnExitQuery;
  end;

type
  TAdvNumberEdit = class(TAdvFocusEdit)
  private
    FBoundsOK: boolean;
    FValueOnEnter: double;
    FBeepOnError: boolean;
    FCanEmpty: boolean;
    FDecimalLen: byte;
    FDecimalSeparator: char;
    FEditorMode: boolean;
    FFunctionList: TFunctionList;
    FInvertSign: boolean;
    FMin: double;
    FMinChecking: boolean;
    FMax: double;
    FMaxChecking: boolean;
    FPrecision: byte;
    FThousandGrouping: boolean;
    FThousandSeparator: char;
    FText: string;
    FValue: double;
    FWholeLen: byte;
    FOnBoundError: TBoundErrorEvent;
    procedure CheckBoundError; // set private variable BoundsOK
    function GetDecimalLength(SetLen: boolean = false) : integer;
    function GetWholeLength(SetLen: boolean = false) : integer;
    procedure MakeNumberMask(var NewPos: integer);
    procedure RefreshText;
    function TextToValue(IncSelStart: boolean = false) : boolean;
    function GetText : string;
    procedure SetText(AValue: string);
  protected
    procedure ErrorBeep; virtual;
    procedure SetCanEmpty(AValue: boolean);
    procedure SetDecimalLen(AValue: byte);
    procedure SetDecimalSeparator(AValue: char);
    procedure SetEditorMode(AValue: boolean);
    procedure SetMin(AValue: double);
    procedure SetMinChecking(AValue: boolean);
    procedure SetMax(AValue: double);
    procedure SetMaxChecking(AValue: boolean);
    procedure SetPrecision(AValue: byte);
    procedure SetThousandGrouping(AValue: boolean);
    procedure SetThousandSeparator(AValue: char);
    procedure SetValue(AValue: double);
    procedure SetWholeLen(AValue: byte);
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure DoEnter; override;
    procedure DoExit; override;
    procedure DoExitQuery(var CanExit: boolean); override;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    procedure WMUndo(var Message: TMessage); message WM_UNDO;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function DecimalLength : integer;
    function WholeLength : integer;
  published
    property CanEmpty: boolean read FCanEmpty write SetCanEmpty default false;
    property BeepOnError: boolean read FBeepOnError write FBeepOnError default true;
    property DecimalLen: byte read FDecimalLen write SetDecimalLen;
    property DecimalSeparator: char read FDecimalSeparator write SetDecimalSeparator default '.';
    property EditorMode: boolean read FEditorMode write SetEditorMode default false;
    property FunctionList: TFunctionList read FFunctionList write FFunctionList;
    property InvertSign: boolean read FInvertSign write FInvertSign default false;
    property Min: double read FMin write SetMin;
    property MinChecking: boolean read FMinChecking write SetMinChecking;
    property Max: double read FMax write SetMax;
    property MaxChecking: boolean read FMaxChecking write SetMaxChecking;
    property Precision: byte read FPrecision write SetPrecision default 0;
    property Text: string read GetText write SetText;
    property ThousandGrouping: boolean read FThousandGrouping write SetThousandGrouping default true;
    property ThousandSeparator: char read FThousandSeparator write SetThousandSeparator default ',';
    property Value: double read FValue write SetValue;
    property WholeLen: byte read FWholeLen write SetWholeLen default 8;
    property OnBoundError: TBoundErrorEvent read FOnBoundError write FOnBoundError;
  end;


function GetMin(a,b: double) : double;
function GetMax(a,b: double) : double;

function MakeItem(Shift: TShiftState; Key: word; Operation: TOperation;
  Value: double) : string;

implementation
uses RTLConsts;

{$R *.res}

function MakeItem(Shift: TShiftState; Key: word;
  Operation: TOperation; Value: double) : string;
var s: string;
begin
  if ssShift in Shift then s:= 'Shift, ' else s:= '';
  if ssCtrl in Shift then s:= s+'Ctrl, ';
  if ssAlt in Shift then s:= s+'Alt, ';
  if s <> '' then SetLength(s, length(s)-2);
  MakeItem:= s+';'+IntToStr(Key)+';'+OprStr[ord(Operation)+1]+';'+
    FloatToStr(Value);
end;

{ TFunctionList }

function TFunctionList.GetKey(Index: integer) : integer;
var s: string;
begin
  if (Index < 0) or (Index > Count) then
    Error(@SListIndexError, Index);
  s:= Strings[Index];
  System.Delete(s,1,pos(';',s));
  Result:= StrToInt(copy(s,1,pos(';',s)-1));
end;

function TFunctionList.GetOperation(Index: integer) : TOperation;
var s: string;
begin
  if (Index < 0) or (Index > Count) then
    Error(@SListIndexError, Index);
  s:= Strings[Index];
  System.Delete(s,1,pos(';',s));
  System.Delete(s,1,pos(';',s));
  Result:= TOperation(pos(s[1],OprStr)-1);
end;

function TFunctionList.GetShift(Index: integer) : TShiftState;
var s: string;
begin
  if (Index < 0) or (Index > Count) then
    Error(@SListIndexError, Index);
  s:= Strings[Index];
  if pos('Shift',s) <> 0 then Result:= [ssShift] else Result:= [];
  if pos('Ctrl',s) <> 0 then Include(Result, ssCtrl);
  if pos('Alt',s) <> 0 then Include(Result, ssAlt);
end;

function TFunctionList.GetValue(Index: integer) : double;
var s: string;
begin
  if (Index < 0) or (Index > Count) then
    Error(@SListIndexError, Index);
  s:= Strings[Index];
  System.Delete(s,1,pos(';',s));
  System.Delete(s,1,pos(';',s)+2);
  Result:= StrToFloat(s);
end;

procedure TFunctionList.InsertNew(Index, ItemNo: integer; NewItem: string);
var
  i,p,sum: integer;
  s,t: string;
begin
  if (Index < 0) or (Index > Count) then
    Error(@SListIndexError, Index);
  if ItemNo <= 0 then exit;
  s:= Strings[Index];
  sum:= 0; t:= s+';';
  repeat
    p:= sum+1; i:= Pos(';',t);
    inc(sum,i); dec(ItemNo);
    if ItemNo > 0 then System.Delete(t,1,i);
  until (ItemNo = 0) or (t = '');
  if t <> '' then
  begin
    System.Delete(s,p,sum-p);
    System.Insert(NewItem,s,p);
    Strings[Index]:= s;
  end;
end;

procedure TFunctionList.SetKey(Index: integer; AValue: integer);
begin
  InsertNew(Index,2,IntToStr(AValue));
end;

procedure TFunctionList.SetOperation(Index: integer; AValue: TOperation);
begin
  InsertNew(Index,3,OprStr[ord(AValue)+1]);
end;

procedure TFunctionList.SetShift(Index: integer; AValue: TShiftState);
var s: string;
begin
  if ssShift in AValue then s:= 'Shift, ' else s:= '';
  if ssCtrl in AValue then s:= s+'Ctrl, ';
  if ssAlt in AValue then s:= s+'Alt, ';
  if s <> '' then SetLength(s, length(s)-2);
  InsertNew(Index, 1, s);
end;

procedure TFunctionList.SetValue(Index: integer; AValue: double);
begin
  InsertNew(Index,4,FloatToStr(AValue));
end;

procedure TFunctionList.GetItem(Index: integer; var Shift: TShiftState;
  var Key: word; var Operation: TOperation; var Value: double);
var
  s: string;
  i: integer;
begin
  if (Index < 0) or (Index > Count) then
    Error(@SListIndexError, Index);
  s:= Strings[Index];
  if pos('Shift',s) <> 0 then Shift:= [ssShift] else Shift:= [];
  if pos('Ctrl',s) <> 0 then Include(Shift, ssCtrl);
  if pos('Alt',s) <> 0 then Include(Shift, ssAlt);
  System.Delete(s,1,pos(';',s)); i:= pos(';',s);
  Key:= StrToInt(copy(s,1,i-1));
  System.Delete(s,1,i);
  Operation:= TOperation(pos(OprStr,s[1]));
  System.Delete(s,1,2);
  Value:= StrToFloat(s);
end;


{ TAdvFocusEdit }

procedure TAdvFocusEdit.CMExit(var Message: TCMExit);
var
  CanExit: boolean;
begin
  CanExit:= true;
  DoExitQuery(CanExit);
  if CanExit then
  begin
    if FOrigParentColor then ParentColor:= true
    else Color:= FOrigColor;
    if FOrigParentFont then ParentFont:= true
    else Font.Assign(FOrigFont);
    DoExit;
  end
  else SetFocus;
end;

procedure TAdvFocusEdit.CreateParams(var Params: TCreateParams);
const
  Alignments: array[TAlignment] of DWORD = (ES_LEFT, ES_RIGHT, ES_CENTER);
begin
  inherited CreateParams(Params);
  with Params do Style:= Style or Alignments[FAlignment];
end;

procedure TAdvFocusEdit.ChangeControl(Back: boolean);
var Msg: TMsg;
begin
  PostMessage(TWinControl(Owner).Handle, WM_NEXTDLGCTL, ord(Back), 0);
  PeekMessage(Msg, Handle, WM_KEYFIRST, WM_KEYLAST, PM_REMOVE);
end;

procedure TAdvFocusEdit.SetAlignment(AValue: TAlignment);
begin
  if FAlignment <> AValue then
  begin
    FAlignment:= AValue;
    RecreateWnd;
  end;
end;

procedure TAdvFocusEdit.SetFocusColor(AValue: TColor);
begin
  if FFocusColor <> AValue then
  begin
    FFocusColor:= AValue;
    if Focused then Color:= AValue;
  end;
end;

procedure TAdvFocusEdit.SetFocusFont(AValue: TFont);
begin
  FFocusFont.Assign(AValue);
  if Focused then Font.Assign(AValue);
end;

procedure TAdvFocusEdit.Click;
begin
  inherited Click;
  if FAutoSelectOnClick and (SelLength = 0) then SelectAll;
end;

procedure TAdvFocusEdit.DoEnter;
begin
  FOrigColor:= Color;
  FOrigFont.Assign(Font);
  FOrigParentColor:= ParentColor;
  FOrigParentFont:= ParentFont;
  Color:= FFocusColor;
  Font.Assign(FFocusFont);
  inherited DoEnter;
end;

procedure TAdvFocusEdit.DoExitQuery(var CanExit: boolean);
begin
  if Assigned(FOnExitQuery) then FOnExitQuery(Self, CanExit);
end;

procedure TAdvFocusEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_RETURN) and ((Shift = []) or (Shift = [ssShift])) and
    FEnterAsTab and (Owner <> nil) then
  begin
    ChangeControl(Shift=[ssShift]);
    Key:= 0; exit;
  end;
  if (Key in [VK_DOWN, VK_UP]) and (Shift = []) and FUpDownAsTab and (Owner <> nil) then
  begin
    ChangeControl(Key=VK_UP);
    Key:= 0; exit;
  end;
  inherited KeyDown(Key, Shift);
end;

constructor TAdvFocusEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAlignment:= taLeftJustify;
  FAutoSelectOnClick:= false;
  FEnterAsTab:= false;
  FUpDownAsTab:= false;
  FOrigFont:= TFont.Create;
  FFocusColor:= clWindow;
  FFocusFont:= TFont.Create;
  FFocusFont.Assign(Font);
end;

destructor TAdvFocusEdit.Destroy;
begin
  FFocusFont.Free;
  FOrigFont.Free;
  inherited Destroy;
end;


{ TAdvNumberEdit }

function GetMin(a,b: double) : double;
begin
  if a < b then GetMin:= a else GetMin:= b;
end;

function GetMax(a,b: double) : double;
begin
  if a > b then GetMax:= a else GetMax:= b;
end;

constructor TAdvNumberEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAlignment:= taRightJustify;
  FBeepOnError:= true;
  FCanEmpty:= false;
  FInvertSign:= false;
  FEditorMode:= false;
  FDecimalLen:= 0;
  FMinChecking:= false;
  FMaxChecking:= false;
  FMin:= 0; FMax:= 0;
  FPrecision:= 0;
  FThousandGrouping:= true;
  FThousandSeparator:= ',';
  FDecimalSeparator:= '.';
  FWholeLen:= 8;
  FText:= '0'; Text:= '0';
  FFunctionList:= TFunctionList.Create;
  FBoundsOK:= true;
end;

destructor TAdvNumberEdit.Destroy;
begin
  FFunctionList.Free;
  inherited Destroy;
end;

procedure TAdvNumberEdit.CheckBoundError;
var
  BoundUnder: boolean;
  ActValue: double;
begin
  if FCanEmpty and (FText = '') then FBoundsOK:= true
  else begin
    ActValue:= GetMin(FMin,FMax);
    BoundUnder:= FMinChecking and (FValue < ActValue);
    if not BoundUnder then ActValue:= GetMax(FMin,FMax);
    if BoundUnder or (FMaxChecking and (FValue > ActValue)) then
    begin
      FBoundsOK:= false;
      if Assigned(FOnBoundError) then FOnBoundError(Self, BoundUnder, ActValue);
      if Visible and Enabled then SetFocus;
      if not Assigned(FOnBoundError) then
        if BoundUnder then raise EBoundError.CreateFmt('Minimum value is %f',[ActValue])
        else raise EBoundError.CreateFmt('Maximum value is %f',[ActValue]);
    end
    else FBoundsOK:= true;
  end;
end;

function TAdvNumberEdit.DecimalLength : integer;
begin
  DecimalLength:= GetDecimalLength;
end;

function TAdvNumberEdit.GetDecimalLength(SetLen: boolean = false) : integer;
var i, Len, LastPos: integer;
begin
  Len:= pos(FDecimalSeparator, FText);
  if Len <> 0 then
  begin
    i:= Len+1; Len:= 0;
    if FDecimalLen = 0 then
    begin
      LastPos:= pos(FDecimalSeparator,FText);
      if LastPos <> 0 then
      begin Len:= FDecimalLen+1; dec(LastPos); end;
    end
    else LastPos:= i;
    while i <= length(FText) do
    begin
      if FText[i] <> FThousandSeparator then inc(Len);
      if Len < FDecimalLen then inc(LastPos);
      inc(i);
    end;
    if SetLen and (Len > FDecimalLen) then
      SetLength(FText, LastPos);
  end;
  GetDecimalLength:= Len;
end;

function TAdvNumberEdit.GetWholeLength(SetLen: boolean = false) : integer;
var i, Len, LastPos: integer;
begin
  i:= 1; Len:= 0; LastPos:= 0;
  while (i <= length(FText)) and (FText[i] <> FDecimalSeparator) do
  begin
    if not (FText[i] in ['-',FThousandSeparator]) then inc(Len);
    if Len <= FWholeLen then inc(LastPos);
    inc(i);
  end;
  if SetLen and (Len > FWholeLen) then
  begin
    i:= pos(FDecimalSeparator, FText); if i = 0 then i:= length(FText)+1;
    Delete(FText,LastPos+1, i-LastPos-1);
  end;
  GetWholeLength:= Len;
end;

procedure TAdvNumberEdit.MakeNumberMask(var NewPos: integer);
var
  i,j,sp: integer;
  FirstDigit: integer;
begin
  i:= 1; sp:= 0;
  if (FText <> '') and (FText[1] = '-') then FirstDigit:= 2
  else FirstDigit:= 1;
  if FText <> '0' then
    while i <= length(FText) do
    begin
      { remove ThousandSeparator and leading 0's chars }
      if (FText[i] = FThousandSeparator) or
        ((i = FirstDigit) and (FText[FirstDigit] = '0')) then
      begin
        if i <= NewPos then dec(sp);
        Delete(FText,i,1); end
      else inc(i);
    end;
  if FText = '' then
    if not CanEmpty then
    begin FText:= '0'; inc(sp); end
    else
  else begin
    if (length(FText) >= FirstDigit) and (FText[FirstDigit] = FDecimalSeparator) then
    begin Insert('0',FText,FirstDigit); inc(sp); end;
    if FThousandGrouping and (not Focused or not FEditorMode) then // add ThousandSeparator chars
    begin
      i:= pos(FDecimalSeparator,FText)-1;
      if i < 0 then i:= length(FText);
      j:= 1; while i > FirstDigit-1 do
      begin
        if (j = 3) and (i <> FirstDigit) then
        begin
          Insert(FThousandSeparator,FText,i);
          j:= 0;
          if i <= NewPos then inc(sp);
        end;
        dec(i); inc(j);
      end;
    end;
  end;
  inc(NewPos,sp);
end;

// In case of KeyPress and KeyDown
procedure TAdvNumberEdit.RefreshText;
var ss, sl: integer;
begin
  if Text <> FText then
  begin
    ss:= SelStart; sl:= SelLength;
    TextToValue;
    SetTextBuf(PChar(FText));
    SelStart:= ss; SelLength:= sl;
  end;
end;

function TAdvNumberEdit.TextToValue(IncSelStart: boolean = false) : boolean;
var
  i: integer;
  v: double;
  ds: char;
  res: boolean;
begin
  ds:= SysUtils.DecimalSeparator; res:= true;
  SysUtils.DecimalSeparator:= FDecimalSeparator;
  i:= 1; while i <= length(FText) do
    if FText[i] = FThousandSeparator then Delete(FText,i,1)
    else inc(i);
  if FCanEmpty and (FText = '') then FValue:= 0
  else if TryStrToFloat(FText, v) then
  begin
    GetWholeLength(true); GetDecimalLength(true);
    FValue:= StrToFloat(FText);
    MakeNumberMask(i);
    if IncSelStart then SelStart:= SelStart+i;
  end
  else res:= false;
  SysUtils.DecimalSeparator:= ds;
  TextToValue:= res;
end;

function TAdvNumberEdit.GetText : string;
var
  Len: Integer;
begin
  Len:= GetTextLen;
  SetString(Result, PChar(nil), Len);
  if Len <> 0 then GetTextBuf(Pointer(Result), Len + 1);
end;

// In case of Text:= xxx
procedure TAdvNumberEdit.SetText(AValue: string);
var
  wText: string;
  i: integer;
begin
  if GetText <> AValue then
  begin
    wText:= FText; FText:= AValue;
     if TextToValue then
    begin
      if (FPrecision <> 0) and (FText <> '') then
      begin
        i:= pos(FDecimalSeparator,FText);
        if i = 0 then
        begin
          FText:= FText+FDecimalSeparator;
          i:= FPrecision;
        end
        else i:= FPrecision-(Length(FText)-i);
        while i > 0 do
        begin
          FText:= FText+'0';
          dec(i);
        end;
      end;
      SetTextBuf(PChar(FText)); // settext, then change event
      CheckBoundError;
    end
    else begin FText:= wText; RefreshText; end;
  end;
end;

procedure TAdvNumberEdit.ErrorBeep;
begin
  if FBeepOnError then MessageBeep(MB_ICONEXCLAMATION);
end;

procedure TAdvNumberEdit.SetCanEmpty(AValue: boolean);
begin
  if FCanEmpty <> AValue then
  begin
    FCanEmpty:= AValue;
    if not FCanEmpty and (FText = '') then
    begin
      if FMinChecking then FText:= FloatToStr(GetMin(FMin,FMax))
      else FText:= '0';
      RefreshText;
    end;
  end;
end;

procedure TAdvNumberEdit.SetDecimalLen(AValue: byte);
begin
  if FDecimalLen <> AValue then
  begin
    FDecimalLen:= AValue;
    GetDecimalLength(true);
    RefreshText;
  end;
end;

procedure TAdvNumberEdit.SetDecimalSeparator(AValue: char);
var i: integer;
begin
  if not (AValue in ['0'..'9', '-', FDecimalSeparator,
    FThousandSeparator]) then
  begin
    i:= pos(FDecimalSeparator, FText);
    if i <> 0 then FText[i]:= AValue;
    FDecimalSeparator:= AValue;
    RefreshText;
  end;
end;

procedure TAdvNumberEdit.SetEditorMode(AValue: boolean);
var i: integer;
begin
  if FEditorMode <> AValue then
  begin
    FEditorMode:= AValue;
    if Focused then MakeNumberMask(i);
  end;
end;

procedure TAdvNumberEdit.SetMin(AValue: double);
var i: double;
begin
  if AValue <> FMin then
  begin
    FMin:= AValue;
    if FCanEmpty and (FText = '') then // OK
    else begin
      i:= GetMin(FMin,FMax);
      if FMinChecking and (FValue < i) then Value:= i;
    end;
  end;
end;

procedure TAdvNumberEdit.SetMinChecking(AValue: boolean);
var i: double;
begin
  if AValue <> FMinChecking then
  begin
    FMinChecking:= AValue;
    if FCanEmpty and (FText = '') then // OK
    begin
      i:= GetMin(FMin, FMax);
      if AValue and (FValue < i) then Value:= i;
    end;
  end;
end;

procedure TAdvNumberEdit.SetMax(AValue: double);
var i: double;
begin
  if AValue <> FMax then
  begin
    FMax:= AValue;
    if FCanEmpty and (FText = '') then // OK
    else begin
      i:= GetMax(FMin,FMax);
      if FMaxChecking and (FValue > i) then Value:= i;
    end;
  end;
end;

procedure TAdvNumberEdit.SetMaxChecking(AValue: boolean);
var i: double;
begin
  if AValue <> FMaxChecking then
  begin
    FMaxChecking:= AValue;
    if FCanEmpty and (FText = '') then // OK
    begin
      i:= GetMax(FMin, FMax);
      if AValue and (FValue > i) then Value:= i;
    end;
  end;
end;

procedure TAdvNumberEdit.SetPrecision(AValue: byte);
begin
  if (FPrecision <> AValue) and (AValue <= FDecimalLen) then
  begin
    FPrecision:= AValue;
    if FPrecision = 0 then FText:= FloatToStr(FValue)
    else FText:= Format('%.*f', [FPrecision, FValue]);
    RefreshText;
  end;
end;

procedure TAdvNumberEdit.SetThousandGrouping(AValue: boolean);
var p: integer;
begin
  if AValue <> FThousandGrouping then
  begin
    FThousandGrouping:= AValue;
    MakeNumberMask(p);
    RefreshText;
  end;
end;

procedure TAdvNumberEdit.SetThousandSeparator(AValue: char);
var i: integer;
begin
  if not (AValue in ['0'..'9', '-', '.', FDecimalSeparator,
    FThousandSeparator]) then
  begin
    if FThousandGrouping then
    repeat
      i:= pos(FThousandSeparator, FText);
      if i <> 0 then FText[i]:= AValue;
    until i = 0;
    FThousandSeparator:= AValue;
    if FThousandGrouping then RefreshText;
  end;
end;

procedure TAdvNumberEdit.SetValue(AValue: double);
var
  ds: char;
begin
  if FValue <> AValue then
  begin
    FValue:= AValue;
    ds:= SysUtils.DecimalSeparator;
    SysUtils.DecimalSeparator:= FDecimalSeparator;
    if FPrecision = 0 then FText:= FloatToStr(FValue)
    else FText:= Format('%.*f', [FPrecision, FValue]);
    SysUtils.DecimalSeparator:= ds;
    RefreshText;
  end;
end;

procedure TAdvNumberEdit.SetWholeLen(AValue: byte);
begin
  if (FWholeLen <> AValue) and (AValue <> 0) then
  begin
    FWholeLen:= AValue;
    GetWholeLength(true);
    RefreshText;
  end;
end;

procedure TAdvNumberEdit.KeyDown(var Key: Word; Shift: TShiftState);
var
  ss,i: integer;
  Msg: TMsg;
  OldValue: double;
begin
  inherited KeyDown(Key, Shift);
  if (Key = 0) or ReadOnly then exit;
  if Key = VK_DELETE then
  begin
    Key:= 0;
    if Shift <> [] then exit;
    ss:= SelStart+1;
    if ss > length(FText) then
    begin
      ErrorBeep;
      exit;
    end;
    if SelLength <> 0 then Delete(FText,ss,SelLength)
    else begin
      if FText[ss] = ThousandSeparator then Delete(FText,ss,1);
      Delete(FText,ss,1);
    end;
    GetWholeLength(true); // Set length, in that case,
    MakeNumberMask(ss);   // if DecimalSeparator char deleted
    i:= ord((FText <> '') and (FText[1] = '-'));
    if ss > i then dec(ss) else ss:= i;
    SelStart:= ss;
    RefreshText;
  end
  else begin
    i:= FunctionList.Count-1;
    while (i >= 0) and ((FunctionList.Shift[i] <> Shift) or
      (FunctionList.Key[i] <> Key)) do dec(i);
    if i >= 0 then
    begin
      OldValue:= FValue;
      case FunctionList.Operation[i] of
        opAdd: Value:= FValue+FunctionList.Value[i];
        opSub: Value:= FValue-FunctionList.Value[i];
        opMul: Value:= FValue*FunctionList.Value[i];
        opDiv: Value:= FValue/FunctionList.Value[i];
      end;
      SelStart:= length(FText); SelLength:= 0;
      PeekMessage(Msg, Handle, WM_KEYFIRST, WM_KEYLAST, PM_REMOVE);
      if OldValue = FValue then ErrorBeep;
    end;
  end;
end;

procedure TAdvNumberEdit.KeyPress(var Key: Char);
var
  ss, DecPos: integer;
  sKey: char;

  function GetShift : boolean;
  var kb: TKeyboardState;
  begin
    GetKeyboardState(kb);
    GetShift:= kb[VK_SHIFT] and $80 <> 0;
  end;

begin
  inherited KeyPress(Key);
  if (Key = #0) or ReadOnly then exit;
  sKey:= Key; Key:= #0;
  if not (sKey in ['-','0'..'9','.',FDecimalSeparator,#8]) then
  begin
    ErrorBeep;
    exit;
  end;
  ss:= SelStart;
  case sKey of
    '-':
    if not FMinChecking or (GetMin(FMin,FMax) < 0) then
      if FInvertSign then
        if (FText = '') or (FText[1] <> '-') then
        begin inc(ss); FText:= '-'+FText; end
        else begin Delete(FText,1,1); dec(ss); end
      else begin
        if SelLength <> 0 then Delete(FText,ss+1,SelLength);
        if (ss = 0) and ((FText = '') or (FText[1] <> '-')) then
        begin
          inc(ss); FText:= '-'+FText;
          MakeNumberMask(ss);
        end
        else ErrorBeep;
      end
    else ErrorBeep;
    '0'..'9':
    begin
      if SelLength <> 0 then Delete(FText,ss+1,SelLength);
      if (ss > 0) or (FText = '') or (FText[1] <> '-') then
      begin
        DecPos:= pos(FDecimalSeparator, FText);
        if (((DecPos = 0) or (ss <= DecPos)) and (WholeLength < FWholeLen)) or
          ((DecPos <> 0) and (ss >= DecPos) and (DecimalLength < FDecimalLen)) then
        begin
          inc(ss);
          Insert(sKey,FText,ss);
          MakeNumberMask(ss);
        end
        else ErrorBeep;
      end
      else ErrorBeep;
    end;
    #8:
    begin
      if SelLength <> 0 then Delete(FText,ss+1,SelLength)
      else if ss <> 0 then
      begin
        if FText[ss] = ThousandSeparator then
        begin Delete(FText,ss,1); dec(ss); end;
        Delete(FText,ss,1);
      end
      else ErrorBeep;
      GetWholeLength(true); // Set length, in that case,
      MakeNumberMask(ss);  // if DecimalSeparator char deleted
      DecPos:= ord((FText <> '') and (FText[1] = '-'));
      if SelLength = 0 then
        if ss > DecPos then dec(ss) else ss:= DecPos;
    end;
    else if (pos(FDecimalSeparator, FText) = 0) and // DecimalSeparator
      (FDecimalLen > 0) then
    begin
      inc(ss);
      Insert(FDecimalSeparator,FText,ss);
      GetDecimalLength(true);
      MakeNumberMask(ss);
    end
    else ErrorBeep;
  end;
  RefreshText; SelStart:= ss;
end;

procedure TAdvNumberEdit.DoEnter;
var i: integer;
begin
  if not FBoundsOK then exit;
  FValueOnEnter:= FValue;
  if FEditorMode then
  begin
    MakeNumberMask(i);
    RefreshText;
  end;
  inherited DoEnter;
end;

procedure TAdvNumberEdit.DoExit;
var i: integer;
begin
  if FEditorMode then MakeNumberMask(i)
  else if (FPrecision <> 0) and (FText <> '') then
    FText:= Format('%.*f', [FPrecision, FValue]);
  RefreshText;
  inherited DoExit;
end;

procedure TAdvNumberEdit.DoExitQuery(var CanExit: boolean);
begin
  CheckBoundError;
  CanExit:= FBoundsOK;
  if CanExit then inherited DoExitQuery(CanExit);
end;

procedure TAdvNumberEdit.WMPaste(var Message: TMessage);
var
  wText: string;
  ss,sl: integer;
begin
  if not ReadOnly and Clipboard.HasFormat(CF_TEXT) then
  begin
    wText:= FText; sl:= SelLength;
    if sl <> 0 then Delete(FText,SelStart+1, sl);
    Insert(Clipboard.AsText,FText, SelStart+1);
    if TextToValue then
    begin
      ss:= SelStart;
      SetTextBuf(PChar(FText));
      SelStart:= ss+Length(FText)-Length(wText)+sl;
    end
    else FText:= wText;
  end;
end;

procedure TAdvNumberEdit.WMUndo(var Message: TMessage);
var Msg: TMsg;
begin
  if not ReadOnly and (FValue <> FValueOnEnter) then
  begin
    Value:= FValueOnEnter;
    // remove unhandled WM_SYSCHAR message from queue
    PeekMessage(Msg, Handle, WM_KEYFIRST, WM_KEYLAST, PM_REMOVE);
  end;
end;

function TAdvNumberEdit.WholeLength : integer;
begin
  WholeLength:= GetWholeLength;
end;

end.

